function [fa_chr, fa_chr_all] = get_replicatesInCellByCelllines(in_loc, ...
    names, cell_nfo, tf, chr)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Read in all peak files from a dir for a given factor and get its 
% cell names info. 
% Put each replicate into a cell, then pool all replicates (cells) 
% under their corresponding cellline (higher level cell). 
%
% Note: 
%  - only ChIP-seq peak centers from replicates are used. 
%  - some cell names have partial overlap, i.e., Sknsh & Sknshra. 
%  - therefore, use cellname + tf name SknshRad21 as search token.
% 
% Output:
%   - fa_chr: peak centers for all cell lines (2-tier cell structure): 
%      - tier1: cell names 
%      - tier2: replicates in the same cell line 
%   - fa_chr_all: peak centers for all cell lines (one vector)
%
% yuanyuan.li@nih.gov
% created : 12-27-2012
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Read in all peak files from the given dir
fa_chr={};
fa_chr_all=[];

fid = fopen(cell_nfo);
cell_names = textscan(fid, '%s\n');
fclose(fid);

peak_names={};
for i=1:numel(names)
    peak_names{i} = names(i).name; 
end
% Read in all factor's ChIP-seq, pool replicates for each cell line 
% and get peak centers for the given chr only for each cell line
for i=1:numel(cell_names{1})
    peaks     = []; % holds tmp peak data
    rep_idx   = 0;  % replicates idx
    fa_chr{i} = {}; % 2-tier cell structure
    
    % Get matched replicates idx
    has_cell = strfind(peak_names, [cell_names{1}{i}, tf]);
    idx = find(not(cellfun('isempty', has_cell)));

    % for each matched replicates, put it into its own cell in tier2, 
    % then into its correspoding celline
    for ii = idx
        %[i, ii]
        rep_idx   = rep_idx+1;
        peaks     = read3ColFile([in_loc, names(ii).name]);
        peaks_chr    = peaks(peaks(:,1)==chr, :);
        peaks_center = ( peaks_chr(:,2)+peaks_chr(:,3) )/2;
        
        fa_chr{i}{rep_idx} = sort(peaks_center, 'ascend'); 
        fa_chr_all = [fa_chr_all; fa_chr{i}{rep_idx}];
        
    end%end-for each matched replicate
end%end-for all celllines

% Sort based on position
fa_chr_all = sort(fa_chr_all, 'ascend');

end % end-function
