function Node = build_kdtree_conRange_noFalsePos(X, dat_chr, threshold)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SYNOPSIS
%   Node = build_kdtree_conRange_noFalsePos(X, leafbound, threshold)
%
% OBJECTIVE
%   builds a kd-tree for a data set
% OUTPUT
%   Node    -  resulting tree
%     each node of the tree contains the following fields:
%       left      -  (tree) left tree
%       right     -  (tree) right tree
%       numpoints -  number of points in node
%       centroid  -  (1xd) mean of points in node
%       cov       -  (dxd) 'covariance' of the points w.r.t. axes centre
%       hyperrect -  (2xd) hyperrectangle bounding the points
%       type      -  'leaf' = node is leaf
%                    'node' = node has 2 children
%       ll        -  (2xk) matrix used to store previously computed 
%                    average log-likelihoods
% INPUT
%   X         -  (dxn) datapoints
%   stop      -  criterion to stop building:
%                grow the full tree until the next level is not
%                constitutive 
%
% yuanyuan.li@nih.gov 
% created: 12-10-2012
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% check the size and the dimension
%
[d,n] = size(X);

%
% error
%
if n==0 
  fprintf('Error: 0 points in node, causing endless loop, press ctrl-C.\n'); 
end

%
%
%
Node.ll   = zeros(2,0);
Node.type = 'node'; % default

%
% if there is 1 datapoint left, make a leaf
%
if n==1
    Node.left      = [];
    Node.right     = [];
    Node.centroid  = X;
    Node.cov       = diag(X*X');
    Node.hyperrect = [X,X];
    Node.type      = 'leaf';
    Node.numpoints = 1;
    return;
end

%
% at least 2 points, calculate the node's easy values
%
Node.numpoints = n;
a              = min(X,[],2); 
b              = max(X,[],2);
Node.hyperrect = [a, b];

%
% no box so leaf
%
if a==b 
  Node.type='leaf'; 
end

%
% Check stopping criteria
%  - if after this split, peaks in either bucket are no longer 
%  - constitutive, then stop here and leave children undefined.
%  - NOTE: assume 1D data
splitpoint = (a+b)/2; 
group_idx = X(1,:)<=splitpoint;
left_bucket = X(:, group_idx);
right_bucket = X(:, ~group_idx);
is_cLeft  = is_conRangeByPctNoFalsePos(dat_chr, left_bucket, threshold);
is_cRight = is_conRangeByPctNoFalsePos(dat_chr, right_bucket, threshold);

if is_cLeft==0 || is_cRight==0
    Node.type = 'leaf';
end


%
% recursively build rest of tree
%
if strcmp(Node.type,'leaf')
  Node.centroid = mean(X,2);
  Node.cov      = diag(X*X')/n;
else
    
  % split box
  % note: splitval is the mean of current level nodes
  [maxwidth, Node.splitdim] = max(b-a); % choose dim with biggest variance
  Node.splitval = (a(Node.splitdim)+b(Node.splitdim))/2; % use mean as spliting val
  inx           = X(Node.splitdim,:)<= Node.splitval; % get idx less than spliting val
  
  eval(['Node.left  = ',mfilename,'(X(:,inx), dat_chr, threshold);']);
  eval(['Node.right = ',mfilename,'(X(:,~inx), dat_chr, threshold);']);
  
  % calc node's values from children's
  % note: centroid is the mean of children
  Node.centroid = (Node.left.centroid*Node.left.numpoints + ...
                   Node.right.centroid*Node.right.numpoints)/n;
  Node.cov      = (Node.left.cov*Node.left.numpoints + ...
                   Node.right.cov*Node.right.numpoints)/n;
end
    
return;
