#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif

#include <math.h>
#include <ctype.h>
#include <sys/types.h>
#include <unistd.h>
#include "gadem.h"
#include "defines.h"
#include "random.h"

int read_userBackgModel(char *fileName,BACKGROUND_Model *back) {

   FILE *fp;
   char *buffer,*tok,*s1;
   int *cn,maxOligomer,len,len2,tabFound,maxAllowedOligomer,order;
   register int ii,i;
   double freq,*sum;

   fp=fopen(fileName,"r");
   if (!fp) { perror(fileName); exit(0); }

   maxAllowedOligomer=MAX_ORDER+1;
   buffer=alloc_char(256);
   s1=alloc_char(maxAllowedOligomer+1);
   sum=alloc_double(maxAllowedOligomer);
   cn=alloc_int(maxAllowedOligomer);
   for (i=0; i<maxAllowedOligomer; i++) { cn[i]=0; sum[i]=0; }

   maxOligomer=0;
   while (!feof(fp)) {
      if ((fgets(buffer,256,fp))>0)  {
         if (buffer[0]!='#') {
            len=strlen(buffer);
            buffer[len-1]='\0';
            tabFound=0;
            for (i=0; i<len; i++) { 
               if (buffer[i]=='\t') { tabFound=1; break; } 
            }
            if (tabFound) {
               tok=strtok(buffer,"\t");
               len2=strlen(tok);
               tok[len2]='\0';
               strcpy(s1,tok);

               if (len2>maxAllowedOligomer) continue;

               if (len2>maxOligomer) maxOligomer=len2;
               tok=strtok(0,"\t");
               freq=atof(tok);
               switch (len2) {
                  case 1:  { back->monomerFreq[cn[0]] =freq; strcpy(back->monomer[cn[0]],s1);  sum[0]+=freq; cn[0]++; break; }
                  case 2:  { back->dimerFreq[cn[1]]   =freq; strcpy(back->dimer[cn[1]],s1);    sum[1]+=freq; cn[1]++; break; }
                  case 3:  { back->trimerFreq[cn[2]]  =freq; strcpy(back->trimer[cn[2]],s1);   sum[2]+=freq; cn[2]++; break; }
                  case 4:  { back->tetramerFreq[cn[3]]=freq; strcpy(back->tetramer[cn[3]],s1); sum[3]+=freq; cn[3]++; break; }
                  case 5:  { back->pentamerFreq[cn[4]]=freq; strcpy(back->pentamer[cn[4]],s1); sum[4]+=freq; cn[4]++; break; }
                  case 6:  { back->hexamerFreq[cn[5]] =freq; strcpy(back->hexamer[cn[5]],s1);  sum[5]+=freq; cn[5]++; break; }
                  case 7:  { back->heptamerFreq[cn[6]]=freq; strcpy(back->heptamer[cn[6]],s1); sum[6]+=freq; cn[6]++; break; }
                  case 8:  { back->octamerFreq[cn[7]] =freq; strcpy(back->octamer[cn[7]],s1);  sum[7]+=freq; cn[7]++; break; }
                  case 9:  { back->nonamerFreq[cn[8]] =freq; strcpy(back->nonamer[cn[8]],s1);  sum[8]+=freq; cn[8]++; break; }
                  default: break; 
               } 
            }
            else {
               tok=strtok(buffer," ");
               len2=strlen(tok);
               if (len2>10) { printf("Error: up to 9th order is allowed!\n"); exit(0); }
               tok=strtok(0," ");
               freq=atof(tok);
               switch (len2) {
                  case 1:  { back->monomerFreq[cn[0]] =freq; sum[0]+=freq; cn[0]++; break; }
                  case 2:  { back->dimerFreq[cn[1]]   =freq; sum[1]+=freq; cn[1]++; break; }
                  case 3:  { back->trimerFreq[cn[2]]  =freq; sum[2]+=freq; cn[2]++; break; }
                  case 4:  { back->tetramerFreq[cn[3]]=freq; sum[3]+=freq; cn[3]++; break; }
                  case 5:  { back->pentamerFreq[cn[4]]=freq; sum[4]+=freq; cn[4]++; break; }
                  case 6:  { back->hexamerFreq[cn[5]] =freq; sum[5]+=freq; cn[5]++; break; }
                  case 7:  { back->heptamerFreq[cn[6]]=freq; sum[6]+=freq; cn[6]++; break; }
                  case 8:  { back->octamerFreq[cn[7]] =freq; sum[7]+=freq; cn[7]++; break; }
                  case 9:  { back->nonamerFreq[cn[8]] =freq; sum[8]+=freq; cn[8]++; break; }
                  default: break; 
               } 
            } 
         } 
      }
   }
   fclose(fp);

   /*---------------------------------------------------------------------------------
   for (i=0; i<4; i++)      printf("%15.14f\n",back->monomerFreq[i]);  printf("\n");
   for (i=0; i<16; i++)     printf("%15.14f\n",back->dimerFreq[i]);    printf("\n");
   for (i=0; i<64; i++)     printf("%15.14f\n",back->trimerFreq[i]);   printf("\n");
   for (i=0; i<256; i++)    printf("%15.14f\n",back->tetramerFreq[i]); printf("\n");
   for (i=0; i<1024; i++)   printf("%15.14f\n",back->pentamerFreq[i]); printf("\n");
   for (i=0; i<4096; i++)   printf("%15.14f\n",back->hexamerFreq[i]);  printf("\n\n");
   for (i=0; i<16384; i++)  printf("%15.14f\n",back->heptamerFreq[i]); printf("\n\n");
   for (i=0; i<65536; i++)  printf("%15.14f\n",back->octamerFreq[i]);  printf("\n\n");
   for (i=0; i<262144; i++) printf("%15.14f\n",back->nonamerFreq[i]);  printf("\n\n");
   ---------------------------------------------------------------------------------*/

   // check probability sum
   for (i=0; i<maxOligomer; i++) {
      if (fabs(sum[i]-1.0)>0.001) printf("sum of marginal %d: %8.6f\n",i+1,sum[i]); 
   }
   if (maxOligomer==0) { printf("Error: no frequencies in %s\n",fileName); exit(0); }

   order=0;
   for (ii=0; ii<maxOligomer; ii++) {
      transition_1st(back->dimerFreq,back->transition1);                   // compute transition from marginal
      order++;
      if (order==maxOligomer-1) break;
      transition_2nd(back->trimerFreq,back->transition2);
      order++;
      if (order==maxOligomer-1) break;
      transition_3rd(back->tetramerFreq,back->transition3);
      order++;
      if (order==maxOligomer-1) break;
      transition_4th(back->pentamerFreq,back->transition4);
      order++;
      if (order==maxOligomer-1) break;
      transition_5th(back->hexamerFreq,back->transition5);
      order++;
      if (order==maxOligomer-1) break;
      transition_6th(back->heptamerFreq,back->transition6);
      order++;
      if (order==maxOligomer-1) break;
      transition_7th(back->octamerFreq,back->transition7);
      order++;
      if (order==maxOligomer-1) break;
      transition_8th(back->nonamerFreq,back->transition8);
      order++;
      if (order==maxOligomer-1) break;
   }

   /*  -------------------------debugging...          ------------------------- 
   for (i=0; i<4; i++)      printf("%8.5f\n",back->monomerFreq[i]);  printf("\n");
   for (i=0; i<16; i++)     printf("%8.5f\n",back->dimerFreq[i]);    printf("\n");
   for (i=0; i<64; i++)     printf("%8.5f\n",back->trimerFreq[i]);   printf("\n");
   for (i=0; i<256; i++)    printf("%8.5f\n",back->tetramerFreq[i]); printf("\n");
   for (i=0; i<1024; i++)   printf("%8.5f\n",back->pentamerFreq[i]); printf("\n");
   for (i=0; i<4096; i++)   printf("%8.5f\n",back->hexamerFreq[i]);  printf("\n\n");
   for (i=0; i<16384; i++)  printf("%8.5f\n",back->heptamerFreq[i]); printf("\n\n");
   for (i=0; i<65536; i++)  printf("%8.5f\n",back->octamerFreq[i]);  printf("\n\n");
   for (i=0; i<262144; i++) printf("%8.5f\n",back->nonamerFreq[i]);  printf("\n\n");
  
   for (i=0; i<16; i++)     printf("%8.5f\n",back->transition1[i]);  printf("\n");
   for (i=0; i<64; i++)     printf("%8.5f\n",back->transition2[i]);  printf("\n");
   for (i=0; i<256; i++)    printf("%8.5f\n",back->transition3[i]);  printf("\n");
   for (i=0; i<1024; i++)   printf("%8.5f\n",back->transition4[i]);  printf("\n");
   for (i=0; i<4096; i++)   printf("%8.5f\n",back->transition5[i]);  printf("\n\n");
   for (i=0; i<16384; i++)  printf("%8.5f\n",back->transition6[i]);  printf("\n\n");
   for (i=0; i<65536; i++)  printf("%8.5f\n",back->transition7[i]);  printf("\n\n");
   for (i=0; i<262144; i++) printf("%8.5f\n",back->transition8[i]);  printf("\n\n");
   exit(0); 
   ---------------------------debugging...          -----------------------------*/

   if (buffer) { free(buffer); buffer=NULL; }
   if (sum)    { free(sum);    sum=NULL;    }
   if (cn)     { free(cn);     cn=NULL;     }
   if (s1)     { free(s1);     s1=NULL;     }

   return (maxOligomer-1);
}

BACKGROUND_Model *alloc_background(void) {
 
   BACKGROUND_Model *back;
   
   back=NULL;

   back=(BACKGROUND_Model *)calloc(1,sizeof(BACKGROUND_Model));

   back->monomerFreq=alloc_double(4+1);       // marginal
   back->dimerFreq=alloc_double(16+1);        // marginal
   back->trimerFreq=alloc_double(64+1);       // marginal
   back->tetramerFreq=alloc_double(256+1);    // marginal
   back->pentamerFreq=alloc_double(1024+1);   // marginal
   back->hexamerFreq=alloc_double(4096+1);    // marginal
   back->heptamerFreq=alloc_double(16384+1);  // marginal
   back->octamerFreq=alloc_double(65536+1);   // marginal
   back->nonamerFreq=alloc_double(262144+1);  // marginal
   //back->decamerFreq=alloc_double(1048576); // marginal
   back->transition1=alloc_double(16+1);      // 1st order
   back->transition2=alloc_double(64+1);      // 2nd order
   back->transition3=alloc_double(256+1);     // 3rd order
   back->transition4=alloc_double(1024+1);    // 4th order
   back->transition5=alloc_double(4096+1);    // 5th order
   back->transition6=alloc_double(16384+1);   // 6th order
   back->transition7=alloc_double(65536+1);   // 7th order
   back->transition8=alloc_double(262144+1);  // 8th order
   //back->transition9=alloc_double(1048576); // 9th order
   back->monomer=alloc_char_char(4,2);       // k-mer
   back->dimer=alloc_char_char(16,3);        // k-mer
   back->trimer=alloc_char_char(64,4);       // k-mer
   back->tetramer=alloc_char_char(256,5);    // k-mer
   back->pentamer=alloc_char_char(1024,6);   // k-mer
   back->hexamer=alloc_char_char(4096,7);    // k-mer
   back->heptamer=alloc_char_char(16384,8);  // k-mer
   back->octamer=alloc_char_char(65536,9);   // k-mer
   back->nonamer=alloc_char_char(262144,10); // k-mer

   return (back);
}

void transition_1st(double *dimerFreq,double *transition1) {

   register int i,j;
   int cn1,cn2;
   double sum;

   /* dimer */
   cn1=0; cn2=0;
   for (i=0; i<4; i++) {
      sum=0; for (j=0; j<4; j++) { sum +=dimerFreq[cn1]; cn1++; }

      if (sum<=PSEUDO_FREQ) {
         for (j=0; j<4; j++) { transition1[cn2]=0.25; cn2++; }
      }
      else {
         for (j=0; j<4; j++)  { 
            transition1[cn2]=(PSEUDO_FREQ+dimerFreq[cn2])/(4*PSEUDO_FREQ+sum);
            cn2++; 
         }
      }
   }
}

void transition_2nd(double *trimerFreq,double *transition2) {

   register int i,k,j;
   int cn1,cn2;
   double sum;

   /* trimer */
   cn1=0; cn2=0;
   for (i=0; i<4; i++) {
      for (k=0; k<4; k++) {
         sum=0; for (j=0; j<4; j++) { sum +=trimerFreq[cn1]; cn1++; }

         if (sum<=PSEUDO_FREQ) {
            for (j=0; j<4; j++) { transition2[cn2]=0.25; cn2++; }
         }
         else {
            for (j=0; j<4; j++)  { 
               transition2[cn2]=(PSEUDO_FREQ+trimerFreq[cn2])/(4*PSEUDO_FREQ+sum);
               cn2++; 
            }
         }
      }
   }
}


void transition_3rd(double *tetramerFreq,double *transition3) {

   register int i,k,l,j;
   int cn1,cn2;
   double sum;

   /* tetramer */
   cn1=0; cn2=0;
   for (i=0; i<4; i++) {
      for (k=0; k<4; k++) {
         for (l=0; l<4; l++) {
            sum=0; for (j=0; j<4; j++) { sum +=tetramerFreq[cn1]; cn1++; }

            if (sum<=PSEUDO_FREQ) {
               for (j=0; j<4; j++) { transition3[cn2]=0.25; cn2++; }
            }
            else {
               for (j=0; j<4; j++)  { 
                  transition3[cn2]=(PSEUDO_FREQ+tetramerFreq[cn2])/(4*PSEUDO_FREQ+sum);
                  cn2++; 
               }
            }
         }
      }
   }
}

void transition_4th(double *pentamerFreq,double *transition4) {

   register int i,k,l,m,j;
   int cn1,cn2;
   double sum;

   /* pentamer */
   cn1=0; cn2=0;
   for (i=0; i<4; i++) {
      for (k=0; k<4; k++) {
         for (l=0; l<4; l++) {
            for (m=0; m<4; m++) {
               sum=0; for (j=0; j<4; j++) { sum +=pentamerFreq[cn1]; cn1++; }

               if (sum<=PSEUDO_FREQ) {
                  for (j=0; j<4; j++) { transition4[cn2]=0.25; cn2++; }
               }
               else {
                  for (j=0; j<4; j++)  { 
                     transition4[cn2]=(PSEUDO_FREQ+pentamerFreq[cn2])/(4*PSEUDO_FREQ+sum);
                     cn2++; 
                  }
               }
            }
         }
      }
   }
}

void transition_5th(double *hexamerFreq,double *transition5) {

   register int i,k,l,m,n,j;
   int cn1,cn2;
   double sum;

   /* hexamer */
   cn1=0; cn2=0;
   for (i=0; i<4; i++) {
      for (k=0; k<4; k++) {
         for (l=0; l<4; l++) {
            for (m=0; m<4; m++) {
               for (n=0; n<4; n++) {
                  sum=0; for (j=0; j<4; j++) { sum +=hexamerFreq[cn1]; cn1++; }
   
                  if (sum<=PSEUDO_FREQ) {
                     for (j=0; j<4; j++) { transition5[cn2]=0.25; cn2++; }
                  }
                  else {
                     for (j=0; j<4; j++)  { 
                        transition5[cn2]=(PSEUDO_FREQ+hexamerFreq[cn2])/(4*PSEUDO_FREQ+sum);
                        cn2++; 
                     }
                  }
               }
            }
         }
      }
   }
}

void transition_6th(double *heptamerFreq,double *transition6) {

   register int i,k,l,m,n,o,j;
   int cn1,cn2;
   double sum;

   /* heptamer */
   cn1=0; cn2=0;
   for (i=0; i<4; i++) {
      for (k=0; k<4; k++) {
         for (l=0; l<4; l++) {
            for (m=0; m<4; m++) {
               for (n=0; n<4; n++) {
                  for (o=0; o<4; o++) {
                     sum=0; for (j=0; j<4; j++) { sum +=heptamerFreq[cn1]; cn1++; }
   
                     if (sum<=PSEUDO_FREQ) {
                        for (j=0; j<4; j++) { transition6[cn2]=0.25; cn2++; }
                     }
                     else {
                        for (j=0; j<4; j++)  { 
                           transition6[cn2]=(PSEUDO_FREQ+heptamerFreq[cn2])/(4*PSEUDO_FREQ+sum);
                           cn2++; 
                        }
                     }
                  }
               }
            }
         }
      }
   }
}

void transition_7th(double *octamerFreq,double *transition7) {

   register int i,k,l,m,n,o,p,j;
   int cn1,cn2;
   double sum;

   /* octamerFreq */
   cn1=0; cn2=0;
   for (i=0; i<4; i++) {
      for (k=0; k<4; k++) {
         for (l=0; l<4; l++) {
            for (m=0; m<4; m++) {
               for (n=0; n<4; n++) {
                  for (o=0; o<4; o++) {
                     for (p=0; p<4; p++) {
                        sum=0; for (j=0; j<4; j++) { sum +=octamerFreq[cn1]; cn1++; }
   
                        if (sum<=PSEUDO_FREQ) {
                           for (j=0; j<4; j++) { transition7[cn2]=0.25; cn2++; }
                        }
                        else {
                           for (j=0; j<4; j++)  { 
                              transition7[cn2]=(PSEUDO_FREQ+octamerFreq[cn2])/(4*PSEUDO_FREQ+sum);
                              cn2++; 
                           }
                        }
                     }
                  }
               }
            }
         }
      }
   }
}

void transition_8th(double *nonamerFreq,double *transition8) {

   register int i,k,l,m,n,o,p,q,j;
   int cn1,cn2;
   double sum;

   /* nonamerFreq */
   cn1=0; cn2=0;
   for (i=0; i<4; i++) {
      for (k=0; k<4; k++) {
         for (l=0; l<4; l++) {
            for (m=0; m<4; m++) {
               for (n=0; n<4; n++) {
                  for (o=0; o<4; o++) {
                     for (p=0; p<4; p++) {
                        for (q=0; q<4; q++) {
                           sum=0; for (j=0; j<4; j++) { sum +=nonamerFreq[cn1]; cn1++; }
   
                           if (sum<=PSEUDO_FREQ) {
                              for (j=0; j<4; j++) { transition8[cn2]=0.25; cn2++; }
                           }
                           else {
                              for (j=0; j<4; j++)  { 
                                 transition8[cn2]=(PSEUDO_FREQ+nonamerFreq[cn2])/(4*PSEUDO_FREQ+sum);
                                 cn2++; 
                              }
                           }
                        }
                     }
                  }
               }
            }
         }
      }
   }
}

void marginal_prob(int *count,int numKmer,double *freq) {

   register int i;
   double sum;

   //printf("number of kmer=%d\n",numKmer);
   //for (i=0; i<numKmer; i++) printf("%d\n",count[i]);

   sum=0; for (i=0; i<numKmer; i++) sum +=(double)count[i];

   if (sum<=PSEUDO_FREQ) {
      printf("Error: data contains no [a,c,g,t].\n");  exit(0);
   }
   else {
      for (i=0; i<numKmer; i++) {
         freq[i]=(PSEUDO_FREQ+(double)count[i])/(sum+numKmer*PSEUDO_FREQ);
      }
   }
}

void simulate_backg_seq_bmodel(BACKGROUND_Model *back,int MarkovOrder,int numSeq,int *seqLen,char **bseq) {

   register int i,j,k;
   int id1,id2,id3,id4,id5,id6,id7,id8,ptr;
   double rand,sum,p;
   // FILE *fp;

   if (MarkovOrder==0) {
      for (i=0; i<numSeq; i++) {
         for (j=0; j<seqLen[i]; j++) { 
            rand=genrand();
            sum=0;
            for (k=0; k<4; k++) {
               p=exp(back->monomerFreq[k]);
               sum+=p;
               if (rand>(sum-p) && rand<=sum) { bseq[i][j]=back->monomer[k][0];  break; }
            }
         } 
         bseq[i][j]='\0';
      }
   }
   else if (MarkovOrder==1) {
      for (i=0; i<numSeq; i++) {
         rand=genrand();
         sum=0;
         for (k=0; k<4; k++) {
            p=exp(back->monomerFreq[k]);
            sum +=p;
            if (rand>(sum-p) && rand<=sum) { bseq[i][0]=back->monomer[k][0]; break; }
         }

         for (j=0; j<seqLen[i]-1; j++) {
            rand=genrand(); id1=0;
            switch (bseq[i][j]) {
               case 'a': id1=0; break; 
               case 'c': id1=1; break; 
               case 'g': id1=2; break; 
               case 't': id1=3; break; 
               default: break;
            }
            sum=0; ptr=id1*4;
            for (k=0; k<4; k++) {
               p=exp(back->transition1[ptr+k]);
               sum+=p;
               if (rand>(sum-p) && rand<=sum) {
                  bseq[i][j+1]=back->dimer[ptr+k][1]; break;
               }
            }
         } 
         bseq[i][seqLen[i]]='\0';
      }
   }
   else if (MarkovOrder==2) {
      for (i=0; i<numSeq; i++) {
         rand=genrand();
         sum=0;
         for (k=0; k<16; k++) {
            p=exp(back->dimerFreq[k]);
            sum+=p;
            if (rand>(sum-p) && rand<=sum) { 
               bseq[i][0]=back->dimer[k][0]; bseq[i][1]=back->dimer[k][1]; break;
            }
         }

         for (j=0; j<seqLen[i]-2; j++) {
            rand=genrand(); id1=-1; id2=-1;
            switch (bseq[i][j]) {
               case 'a': id1=0; break; 
               case 'c': id1=1; break; 
               case 'g': id1=2; break; 
               case 't': id1=3; break; 
               default: break;
            }
            switch (bseq[i][j+1]) {
               case 'a': id2=0; break; 
               case 'c': id2=1; break; 
               case 'g': id2=2; break; 
               case 't': id2=3; break; 
               default: break;
            }
            sum=0; ptr=id1*16+id2*4;
            for (k=0; k<4; k++) {
               p=exp(back->transition2[ptr+k]);
               sum+=p;
               if (rand>(sum-p) && rand<=sum) {
                  bseq[i][j+2]=back->trimer[ptr+k][2]; break;
               }
            }
         } 
         bseq[i][seqLen[i]]='\0';
      }
   }
   else if (MarkovOrder==3) {
      for (i=0; i<numSeq; i++) {
         rand=genrand();
         sum=0;
         for (k=0; k<64; k++) {
            p=exp(back->trimerFreq[k]);
            sum+=p;
            if (rand>(sum-p) && rand<=sum) { 
               bseq[i][0]=back->trimer[k][0]; bseq[i][1]=back->trimer[k][1]; bseq[i][2]=back->trimer[k][2]; break;
            }
         }
         for (j=0; j<seqLen[i]-3; j++) {
            rand=genrand(); id1=-1; id2=-1; id3=-1;
            switch (bseq[i][j]) {
               case 'a': id1=0; break; 
               case 'c': id1=1; break; 
               case 'g': id1=2; break; 
               case 't': id1=3; break; 
               default: break;
            }
            switch (bseq[i][j+1]) {
               case 'a': id2=0; break; 
               case 'c': id2=1; break; 
               case 'g': id2=2; break; 
               case 't': id2=3; break; 
               default: break;
            }
            switch (bseq[i][j+2]) {
               case 'a': id3=0; break; 
               case 'c': id3=1; break; 
               case 'g': id3=2; break; 
               case 't': id3=3; break; 
               default: break;
            }
            sum=0; ptr=id1*64+id2*16+id3*4;
            for (k=0; k<4; k++) {
               p=exp(back->transition3[ptr+k]);
               sum+=p;
               if (rand>(sum-p) && rand<=sum) {
                  bseq[i][j+3]=back->tetramer[ptr+k][3]; break;
               }
            }
         } bseq[i][seqLen[i]]='\0';
      }
   }
   else if (MarkovOrder==4) {
      for (i=0; i<numSeq; i++) {
         rand=genrand();
         sum=0;
         for (k=0; k<256; k++) {
            p=exp(back->tetramerFreq[k]);
            sum+=p;
            if (rand>(sum-p) && rand<=sum) { 
               bseq[i][0]=back->tetramer[k][0]; bseq[i][1]=back->tetramer[k][1]; 
               bseq[i][2]=back->tetramer[k][2]; bseq[i][3]=back->tetramer[k][3]; break;
            }
         }
         for (j=0; j<seqLen[i]-4; j++) {
            rand=genrand(); id1=-1; id2=-1; id3=-1; id4=-1;
            switch (bseq[i][j]) {
               case 'a': id1=0; break; 
               case 'c': id1=1; break; 
               case 'g': id1=2; break; 
               case 't': id1=3; break; 
               default: break;
            }
            switch (bseq[i][j+1]) {
               case 'a': id2=0; break; 
               case 'c': id2=1; break; 
               case 'g': id2=2; break; 
               case 't': id2=3; break; 
               default: break;
            }
            switch (bseq[i][j+2]) {
               case 'a': id3=0; break; 
               case 'c': id3=1; break; 
               case 'g': id3=2; break; 
               case 't': id3=3; break; 
               default: break;
            }
            switch (bseq[i][j+3]) {
               case 'a': id4=0; break; 
               case 'c': id4=1; break; 
               case 'g': id4=2; break; 
               case 't': id4=3; break; 
               default: break;
            }
            sum=0; ptr=id1*256+id2*64+id3*16+id4*4;
            for (k=0; k<4; k++) {
               p=exp(back->transition4[ptr+k]);
               sum+=p;
               if (rand>(sum-p) && rand<=sum) {
                  bseq[i][j+4]=back->pentamer[ptr+k][4]; break;
               }
            }
         } bseq[i][seqLen[i]]='\0';
      }
   }
   else if (MarkovOrder==5) {
      for (i=0; i<numSeq; i++) {
         rand=genrand();
         sum=0;
         for (k=0; k<1024; k++) {
            p=exp(back->pentamerFreq[k]);
            sum+=p;
            if (rand>(sum-p) && rand<=sum) { 
               bseq[i][0]=back->pentamer[k][0]; bseq[i][1]=back->pentamer[k][1]; bseq[i][2]=back->pentamer[k][2]; 
               bseq[i][3]=back->pentamer[k][3]; bseq[i][4]=back->pentamer[k][4]; break;
            }
         }
         for (j=0; j<seqLen[i]-5; j++) {
            rand=genrand(); id1=-1; id2=-1; id3=-1; id4=-1; id5=-1;
            switch (bseq[i][j]) {
               case 'a': id1=0; break; 
               case 'c': id1=1; break; 
               case 'g': id1=2; break; 
               case 't': id1=3; break; 
               default: break;
            }
            switch (bseq[i][j+1]) {
               case 'a': id2=0; break; 
               case 'c': id2=1; break; 
               case 'g': id2=2; break; 
               case 't': id2=3; break; 
               default: break;
            }
            switch (bseq[i][j+2]) {
               case 'a': id3=0; break; 
               case 'c': id3=1; break; 
               case 'g': id3=2; break; 
               case 't': id3=3; break; 
               default: break;
            }
            switch (bseq[i][j+3]) {
               case 'a': id4=0; break; 
               case 'c': id4=1; break; 
               case 'g': id4=2; break; 
               case 't': id4=3; break; 
               default: break;
            }
            switch (bseq[i][j+4]) {
               case 'a': id5=0; break; 
               case 'c': id5=1; break; 
               case 'g': id5=2; break; 
               case 't': id5=3; break; 
               default: break;
            }
            sum=0; ptr=id1*1024+id2*256+id3*64+id4*16+id5*4;
            for (k=0; k<4; k++) {
               p=exp(back->transition5[ptr+k]);
               sum+=p;
               if (rand>(sum-p) && rand<=sum) {
                  bseq[i][j+5]=back->hexamer[ptr+k][5]; break;
               }
            }
         } bseq[i][seqLen[i]]='\0';
      }
   }
   else if (MarkovOrder==6) {
      for (i=0; i<numSeq; i++) {
         rand=genrand();
         sum=0;
         for (k=0; k<4096; k++) {
            p=exp(back->hexamerFreq[k]);
            sum+=p;
            if (rand>(sum-p) && rand<=sum) { 
               bseq[i][0]=back->hexamer[k][0]; bseq[i][1]=back->hexamer[k][1]; bseq[i][2]=back->hexamer[k][2]; 
               bseq[i][3]=back->hexamer[k][3]; bseq[i][4]=back->hexamer[k][4]; bseq[i][5]=back->hexamer[k][5]; break;
            }
         }
         for (j=0; j<seqLen[i]-6; j++) {
            rand=genrand(); id1=-1; id2=-1; id3=-1; id4=-1; id5=-1; id6=-1;
            switch (bseq[i][j]) {
               case 'a': id1=0; break; 
               case 'c': id1=1; break; 
               case 'g': id1=2; break; 
               case 't': id1=3; break; 
               default: break;
            }
            switch (bseq[i][j+1]) {
               case 'a': id2=0; break; 
               case 'c': id2=1; break; 
               case 'g': id2=2; break; 
               case 't': id2=3; break; 
               default: break;
            }
            switch (bseq[i][j+2]) {
               case 'a': id3=0; break; 
               case 'c': id3=1; break; 
               case 'g': id3=2; break; 
               case 't': id3=3; break; 
               default: break;
            }
            switch (bseq[i][j+3]) {
               case 'a': id4=0; break; 
               case 'c': id4=1; break; 
               case 'g': id4=2; break; 
               case 't': id4=3; break; 
               default: break;
            }
            switch (bseq[i][j+4]) {
               case 'a': id5=0; break; 
               case 'c': id5=1; break; 
               case 'g': id5=2; break; 
               case 't': id5=3; break; 
               default: break;
            }
            switch (bseq[i][j+5]) {
               case 'a': id6=0; break; 
               case 'c': id6=1; break; 
               case 'g': id6=2; break; 
               case 't': id6=3; break; 
               default: break;
            }
            sum=0; ptr=id1*4096+id2*1024+id3*256+id4*64+id5*16+id6*4;
            for (k=0; k<4; k++) {
               p=exp(back->transition6[ptr+k]);
               sum+=p;
               if (rand>(sum-p) && rand<=sum) {
                  bseq[i][j+6]=back->heptamer[ptr+k][6]; break;
               }
            }
         } bseq[i][seqLen[i]]='\0';
      }
   }
   else if (MarkovOrder==7) {
      for (i=0; i<numSeq; i++) {
         rand=genrand();
         sum=0;
         for (k=0; k<16384; k++) {
            p=exp(back->heptamerFreq[k]);
            sum+=p;
            if (rand>(sum-p) && rand<=sum) { 
               bseq[i][0]=back->heptamer[k][0]; bseq[i][1]=back->heptamer[k][1]; bseq[i][2]=back->heptamer[k][2]; bseq[i][3]=back->heptamer[k][3]; 
               bseq[i][4]=back->heptamer[k][4]; bseq[i][5]=back->heptamer[k][5]; bseq[i][6]=back->heptamer[k][6]; break;
            }
         }
         for (j=0; j<seqLen[i]-7; j++) {
            rand=genrand(); id1=-1; id2=-1; id3=-1; id4=-1; id5=-1; id6=-1; id7=-1;
            switch (bseq[i][j]) {
               case 'a': id1=0; break; 
               case 'c': id1=1; break; 
               case 'g': id1=2; break; 
               case 't': id1=3; break; 
               default: break;
            }
            switch (bseq[i][j+1]) {
               case 'a': id2=0; break; 
               case 'c': id2=1; break; 
               case 'g': id2=2; break; 
               case 't': id2=3; break; 
               default: break;
            }
            switch (bseq[i][j+2]) {
               case 'a': id3=0; break; 
               case 'c': id3=1; break; 
               case 'g': id3=2; break; 
               case 't': id3=3; break; 
               default: break;
            }
            switch (bseq[i][j+3]) {
               case 'a': id4=0; break; 
               case 'c': id4=1; break; 
               case 'g': id4=2; break; 
               case 't': id4=3; break; 
               default: break;
            }
            switch (bseq[i][j+4]) {
               case 'a': id5=0; break; 
               case 'c': id5=1; break; 
               case 'g': id5=2; break; 
               case 't': id5=3; break; 
               default: break;
            }
            switch (bseq[i][j+5]) {
               case 'a': id6=0; break; 
               case 'c': id6=1; break; 
               case 'g': id6=2; break; 
               case 't': id6=3; break; 
               default: break;
            }
            switch (bseq[i][j+6]) {
               case 'a': id7=0; break; 
               case 'c': id7=1; break; 
               case 'g': id7=2; break; 
               case 't': id7=3; break; 
               default: break;
            }
            sum=0; ptr=id1*16384+id2*4096+id3*1024+id4*256+id5*64+id6*16+id7*4;
            for (k=0; k<4; k++) {
               p=exp(back->transition7[ptr+k]);
               sum+=p;
               if (rand>(sum-p) && rand<=sum) {
                  bseq[i][j+7]=back->octamer[ptr+k][7]; break;
               }
            }
         } bseq[i][seqLen[i]]='\0';
      }
   }
   else if (MarkovOrder==8) {
      for (i=0; i<numSeq; i++) {
         rand=genrand();
         sum=0;
         for (k=0; k<65536; k++) {
            p=exp(back->octamerFreq[k]);
            sum+=p;
            if (rand>(sum-p) && rand<=sum) { 
               bseq[i][0]=back->octamer[k][0]; bseq[i][1]=back->octamer[k][1]; bseq[i][2]=back->octamer[k][2]; bseq[i][3]=back->octamer[k][3]; 
               bseq[i][4]=back->octamer[k][4]; bseq[i][5]=back->octamer[k][5]; bseq[i][6]=back->octamer[k][6]; bseq[i][7]=back->octamer[k][7]; break;
            }
         }
         for (j=0; j<seqLen[i]-8; j++) {
            rand=genrand(); id1=-1; id2=-1; id3=-1; id4=-1; id5=-1; id6=-1; id7=-1; id8=-1;
            switch (bseq[i][j]) {
               case 'a': id1=0; break; 
               case 'c': id1=1; break; 
               case 'g': id1=2; break; 
               case 't': id1=3; break; 
               default: break;
            }
            switch (bseq[i][j+1]) {
               case 'a': id2=0; break; 
               case 'c': id2=1; break; 
               case 'g': id2=2; break; 
               case 't': id2=3; break; 
               default: break;
            }
            switch (bseq[i][j+2]) {
               case 'a': id3=0; break; 
               case 'c': id3=1; break; 
               case 'g': id3=2; break; 
               case 't': id3=3; break; 
               default: break;
            }
            switch (bseq[i][j+3]) {
               case 'a': id4=0; break; 
               case 'c': id4=1; break; 
               case 'g': id4=2; break; 
               case 't': id4=3; break; 
               default: break;
            }
            switch (bseq[i][j+4]) {
               case 'a': id5=0; break; 
               case 'c': id5=1; break; 
               case 'g': id5=2; break; 
               case 't': id5=3; break; 
               default: break;
            }
            switch (bseq[i][j+5]) {
               case 'a': id6=0; break; 
               case 'c': id6=1; break; 
               case 'g': id6=2; break; 
               case 't': id6=3; break; 
               default: break;
            }
            switch (bseq[i][j+6]) {
               case 'a': id7=0; break; 
               case 'c': id7=1; break; 
               case 'g': id7=2; break; 
               case 't': id7=3; break; 
               default: break;
            }
            switch (bseq[i][j+7]) {
               case 'a': id8=0; break; 
               case 'c': id8=1; break; 
               case 'g': id8=2; break; 
               case 't': id8=3; break; 
               default: break;
            }
            sum=0; ptr=id1*65536+id2*16384+id3*4096+id4*1024+id5*256+id6*64+id7*16+id8*4;
            for (k=0; k<4; k++) {
               p=exp(back->transition8[ptr+k]);
               sum+=p;
               if (rand>(sum-p) && rand<=sum) {
                  bseq[i][j+8]=back->nonamer[ptr+k][8]; break;
               }
            }
         } bseq[i][seqLen[i]]='\0';
      }
   }
   else { printf("Error: max order: 8\n"); exit(0); }

   /*-------------------------------------------------
   fp=fopen("simulated_0th.seq","w");
   for (i=0; i<numSeq; i++) {
      fprintf(fp,">test\n");
      fprintf(fp,"%s\n",bseq[i]); 
   } fclose(fp);
   -------------------------------------------------*/
}

