//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
//ART_454 -- Artificial Read Transcriber 
//Copyright(c) 2008-2011 Weichun Huang, All Rights Reserved.
//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
#pragma once
#include <iostream>
#include <vector>
#include <sstream>
#include <fstream>
#include <string>
#include <map>
#include <cstdlib>
#include <climits>

#define IS_CUM_DIST false
#define HIGHEST_QUAL 80
//QUAL_DIST_ONE len
//QUAL_DIST_TWO len pos pre_qual
#define QUAL_DIST_1BASE "1	0	2	3	4	5	6	7	8	9	10	11	12	13	14	15	16	17	18	19	20	21	22	23	24	25	26	27	28	29\n1	22087	85	2579	11536	20551	26545	28446	26695	45654	36660	30935	64766	57117	59947	79074	198445	374207	373069	439749	989206	1067501	1598774	3318178	3781533	7839932	14187500	38833367	37936667	10426\n2	16	17	18	19	20	21	22	23	24	25	26	27	28	29	30	31	32	33	34	35	36	37\n2	491	1130	16689	22522	30171	28911	51558	47136	98321	90751	192399	186012	368430	586811	588310	1733910	2478018	4580801	11223635	4159818	2168693	2869039\n3	28	29	30	31	32	33	34	35	36	37	38	39	40	41\n3	4034	3874	21018	79979	107070	228124	392590	635317	1252518	3194994	159629	244464	579381	709414\n4	31	32	33	34	35	36	37	38	39	40	41	42	43\n4	4	2700	15076	43639	380	167585	327656	971117	41461	40078	81569	143451	241301\n5	31	33	34	35	38	39	43	44\n5	3	8174	8525	52	340180	72501	45545	93425\n6	32	33	34	37	38	39	44	45\n6	1095	2531	4	11	89407	3005	31612	11\n7	30	31	32	33	37	38	39	44	45\n7	1	388	671	2	2775	23559	27	5544	3599\n8	28	29	30	31	36	37	38	44	45\n8	1	16	153	117	324	6342	621	1941	535\n9	28	29	30	35	36	37	44\n9	30	68	3	98	1854	380	813\n10	26	27	28	34	35	36	43	44\n10	1	13	15	29	592	109	46	190\n11	25	26	27	33	34	35	43\n11	2	9	1	9	228	19	66\n12	25	33	34	43\n12	3	67	19	29\n13	23	24	32	33	43\n13	2	1	3	33	10\n14	23	32	33	43\n14	1	11	2	2\n15	32	43\n15	5	3\n16	32\n16	1\n17	33\n17	1\n18	33\n18	1\n"
#define QUAL_DIST_MC "2	1	16	2	3	4\n2	1	16	68	139	284\n2	1	17	2	3	4	5	6\n2	1	17	28	64	107	409	522\n2	1	18	4	5	6	7	8\n2	1	18	3066	7830	4872	597	324\n2	1	19	5	6	7	8	9\n2	1	19	2937	5692	12690	407	796\n2	1	20	5	6	8	9	10\n2	1	20	1696	3793	15048	8653	981\n2	1	21	7	8	9	10	11	12\n2	1	21	4522	2519	9276	10684	628	1282\n2	1	22	8	9	10	11	12	13	14\n2	1	22	2670	2970	3159	25133	15105	1552	969\n2	1	23	10	11	12	13	14	15\n2	1	23	3536	3909	17383	20132	1040	1136\n2	1	24	1	2	11	12	13	14	15	16\n2	1	24	4	1	4544	11101	22552	55715	1246	3158\n2	1	25	1	2	3	4	13	15	16	17\n2	1	25	2	1	4	5	14071	72272	239	4157\n2	1	26	1	2	3	4	5	6	14	15	16	17	18	19\n2	1	26	10	57	78	41	3	1	8651	21003	44312	109551	5439	3253\n2	1	27	2	3	4	5	6	7	8	16	17	18	19	20\n2	1	27	3	9	31	103	40	12	5	27705	1	144712	4231	9160\n2	1	28	5	6	7	8	9	10	17	19	20	21	22	24	25\n2	1	28	5	74	148	6	9	6	37039	191666	117880	12168	7564	348	1517\n2	1	29	6	7	8	9	10	11	18	19	20	21	22	23	25	26\n2	1	29	4	28	195	188	16	25	48661	29966	136209	337484	18768	11678	2835	754\n2	1	30	8	9	10	11	12	13	17	18	19	20	21	22	23	24	25	26\n2	1	30	37	199	506	147	48	54	1	1	34764	40060	207429	238783	13214	32599	106	20362\n2	1	31	10	11	12	13	14	15	17	20	21	22	23	24	25	26	27\n2	1	31	126	843	980	33	142	118	13	46157	114858	269280	662077	417415	76991	130174	14703\n2	1	32	11	12	13	14	15	16	17	18	22	23	24	25	26	27	28\n2	1	32	150	780	2003	1845	56	118	367	1	151358	92286	948475	555592	211512	409216	104259\n2	1	33	13	14	15	16	17	23	24	25	26	27	28\n2	1	33	605	1913	5022	5282	2909	224722	134760	1284687	2338681	361635	220585\n2	1	34	14	15	16	17	24	25	26	27	28\n2	1	34	420	1508	2319	1110	151241	356258	863538	4542375	5304866\n2	1	35	25	26	27	28\n2	1	35	205877	725843	52044	3176054\n2	1	36	27	28	29\n2	1	36	1461483	633820	73390\n2	1	37	28	29\n2	1	37	1798955	1070084\n3	1	28	24	25\n3	1	28	1154	2880\n3	1	29	25	26	27\n3	1	29	81	3340	453\n3	1	30	17	18	25	26	27	28\n3	1	30	2	8	183	15299	4412	1114\n3	1	31	18	19	20	25	26	27	28	29\n3	1	31	2	7	20	2	50495	18683	8422	2348\n3	1	32	17	18	19	20	21	22	23	27	28	29	30\n3	1	32	120	239	43	9	25	23	1	51117	35369	11968	8156\n3	1	33	17	18	19	20	22	26	27	28	29	30	31\n3	1	33	19	51	326	323	1	5841	1531	68427	95589	25855	30161\n3	1	34	18	19	20	21	22	23	27	28	29	30	31	32\n3	1	34	25	103	223	813	949	49	14639	3112	83919	163618	51204	73936\n3	1	35	20	21	22	23	28	29	30	31\n3	1	35	114	219	396	103	22242	6271	138288	467684\n3	1	36	21	22	23	29	30	31	32	33\n3	1	36	59	387	200	33048	11040	118380	1075036	14368\n3	1	37	30	31	32	33	34\n3	1	37	56936	18488	163846	2955643	81\n3	1	38	31	32	33	34\n3	1	38	94765	61169	856	2839\n3	1	39	32	33\n3	1	39	148359	96105\n3	1	40	33	34\n3	1	40	274042	305339\n3	1	41	34\n3	1	41	709414\n3	2	17	1	2\n3	2	17	136	5\n3	2	18	1	2	3	4\n3	2	18	3	160	160	2\n3	2	19	3	4	5\n3	2	19	41	246	192\n3	2	20	5	6	7\n3	2	20	195	491	3\n3	2	21	6	7	8	9\n3	2	21	3	573	538	2\n3	2	22	8	9	10	11	12\n3	2	22	175	795	574	209	3\n3	2	23	10	11	12\n3	2	23	67	255	31\n3	2	24	1	2\n3	2	24	588	566\n3	2	25	1	2	3	4	5\n3	2	25	279	556	1114	1133	64\n3	2	26	1	2	3	4	5	6	7\n3	2	26	2802	28718	24677	15445	1820	1410	103\n3	2	27	2	3	4	5	6	7	8\n3	2	27	1531	7202	16183	46096	15570	2712	1541\n3	2	28	5	6	7	8	9	10\n3	2	28	14948	45611	68539	2347	5160	2081\n3	2	29	6	7	8	9	10	11\n3	2	29	6271	23466	105418	83699	5538	8751\n3	2	30	8	9	10	11	12	13	17	18\n3	2	30	22892	94561	218829	36881	15559	14754	373	44\n3	2	31	10	11	12	13	14	15	16	17	18\n3	2	31	61777	311299	326233	8669	34271	29971	996	7018	448\n3	2	32	11	12	13	14	15	16	17	18\n3	2	32	61169	272371	565968	475291	14047	41349	92039	112\n3	2	33	12	13	14	15	16	17	18\n3	2	33	4683	191228	492536	1084788	1046137	521580	62\n3	2	34	14	15	16	17\n3	2	34	110447	318051	395210	193965\n4	1	31	22	23	24\n4	1	31	1	2	1\n4	1	32	22	23	30\n4	1	32	3	11	2686\n4	1	33	30	31	32\n4	1	33	571	13529	976\n4	1	34	22	23	24	32	33\n4	1	34	33	401	261	40863	2081\n4	1	35	24	33\n4	1	35	24	356\n4	1	36	23	24	32	33\n4	1	36	119	106	131095	36265\n4	1	37	33	34\n4	1	37	245923	81733\n4	1	38	34	35\n4	1	38	762867	208250\n4	1	39	35\n4	1	39	41461\n4	1	40	33	34\n4	1	40	26828	13250\n4	1	41	34	35\n4	1	41	64413	17156\n4	1	42	35	36\n4	1	42	143428	23\n4	1	43	36\n4	1	43	241301\n4	2	22	11	12\n4	2	22	6	31\n4	2	23	12	13	14	15\n4	2	23	162	224	129	18\n4	2	24	13	14	15\n4	2	24	1	150	241\n4	2	30	17	18	19\n4	2	30	577	2582	98\n4	2	31	17	18	19	20	22\n4	2	31	32	941	5635	6845	76\n4	2	32	17	18	19	20	21	22	23\n4	2	32	26162	87464	17469	3788	18302	19309	440\n4	2	33	17	18	19	20	22	23\n4	2	33	7321	19691	118833	163171	1845	592\n4	2	34	18	19	20	21	22	23\n4	2	34	9351	45896	103896	384114	346438	32568\n4	2	35	20	21	22	23	24\n4	2	35	56447	104137	228061	21649	1\n4	2	36	21	22	23\n4	2	36	23593	166522	51209\n4	3	11	1\n4	3	11	6\n4	3	12	1	2\n4	3	12	122	71\n4	3	13	2	3	4\n4	3	13	67	141	17\n4	3	14	3	4	5	6\n4	3	14	10	129	139	1\n4	3	15	5	6	7	8\n4	3	15	29	149	77	4\n4	3	17	1	2\n4	3	17	31355	2737\n4	3	18	1	2	3	4\n4	3	18	972	53053	65896	108\n4	3	19	2	3	4	5\n4	3	19	6	16509	103645	67771\n4	3	20	4	5	6	7\n4	3	20	205	99848	223633	10461\n4	3	21	6	7	8	9\n4	3	21	3035	262133	264341	637\n4	3	22	8	9	10	11	12\n4	3	22	76994	359714	265148	58851	1544\n4	3	23	9	10	11	12\n4	3	23	18	18047	81268	7125\n4	3	24	12\n4	3	24	1\n5	1	31	23\n5	1	31	3\n5	1	33	23	24	31	32\n5	1	33	28	39	3155	4952\n5	1	34	23	24	32	33\n5	1	34	16	47	8458	4\n5	1	35	24\n5	1	35	52\n5	1	38	33	34	35\n5	1	38	62	340099	19\n5	1	39	34	35\n5	1	39	67416	5085\n5	1	43	35	36\n5	1	43	4671	40874\n5	1	44	35	36\n5	1	44	15111	78314\n5	2	23	15	16	17\n5	2	23	19	25	3\n5	2	24	15	16	17\n5	2	24	14	106	18\n5	2	31	22	23	24\n5	2	31	1086	2036	33\n5	2	32	22	23	24\n5	2	32	891	11252	1267\n5	2	33	23\n5	2	33	66\n5	2	34	22	23	24\n5	2	34	5956	226226	175333\n5	2	35	23	24\n5	2	35	4770	20116\n5	2	36	23	24\n5	2	36	60485	58703\n5	3	15	7	8\n5	3	15	10	23\n5	3	16	8	9	10	11\n5	3	16	33	51	46	1\n5	3	17	10	11\n5	3	17	7	14\n5	3	22	10	11	12	13\n5	3	22	3	1643	6189	98\n5	3	23	11	12	13	14	15\n5	3	23	107	71790	132837	95223	4878\n5	3	24	13	14	15	16\n5	3	24	54	95123	154342	5933\n5	4	7	1\n5	4	7	10\n5	4	8	1	2\n5	4	8	38	18\n5	4	9	1	2	3\n5	4	9	2	26	23\n5	4	10	1	3	4	5\n5	4	10	3	8	38	7\n5	4	11	1	2	5\n5	4	11	1738	12	15\n5	4	12	1	2	3\n5	4	12	46162	31530	287\n5	4	13	1	2	3	4	5\n5	4	13	64	39472	84550	8880	23\n5	4	14	2	3	4	5	6	7\n5	4	14	7	7032	100595	81781	913	18\n5	4	15	4	5	6	7	8\n5	4	15	30	28804	85912	41908	2566\n5	4	16	6	7	8	9\n5	4	16	7	3810	2113	3\n6	1	32	22	23	30	31\n6	1	32	1	10	562	522\n6	1	33	23	24	31	32\n6	1	33	12	3	2501	15\n6	1	34	23\n6	1	34	4\n6	1	37	32\n6	1	37	11\n6	1	38	32	33	34\n6	1	38	10	44603	44794\n6	1	39	33	34\n6	1	39	9	2996\n6	1	44	34	35	36\n6	1	44	4696	26914	2\n6	1	45	35\n6	1	45	11\n6	2	22	16\n6	2	22	1\n6	2	23	16	17\n6	2	23	10	16\n6	2	24	17\n6	2	24	3\n6	2	30	22	23\n6	2	30	3	559\n6	2	31	23	24\n6	2	31	2624	399\n6	2	32	23	24\n6	2	32	21	15\n6	2	33	23	24\n6	2	33	13145	31467\n6	2	34	23	24\n6	2	34	5216	47270\n6	2	35	24\n6	2	35	26925\n6	2	36	24\n6	2	36	2\n6	3	16	11	12\n6	3	16	9	2\n6	3	17	11	12\n6	3	17	17	2\n6	3	22	16\n6	3	22	3\n6	3	23	14	15	16	17\n6	3	23	6	6883	12865	1811\n6	3	24	15	16	17\n6	3	24	6991	77609	21478\n6	4	11	6	7\n6	4	11	19	7\n6	4	12	7	8\n6	4	12	1	3\n6	4	14	6	7\n6	4	14	2	4\n6	4	15	7	8	9\n6	4	15	4104	9685	85\n6	4	16	7	8	9	10	11\n6	4	16	55	22314	40268	26660	1180\n6	4	17	9	10	11	12\n6	4	17	22	11391	11870	6\n6	5	6	1\n6	5	6	21\n6	5	7	1	2\n6	5	7	4162	9\n6	5	8	1	2	3\n6	5	8	22586	9395	21\n6	5	9	1	2	3	4\n6	5	9	1508	21204	17247	416\n6	5	10	2	3	4	5	6\n6	5	10	81	11749	20606	5602	13\n6	5	11	3	4	5	6\n6	5	11	2	1730	10272	1046\n6	5	12	5	6\n6	5	12	2	4\n7	1	30	28\n7	1	30	1\n7	1	31	22	28	29	30\n7	1	31	1	5	374	8\n7	1	32	29	30	31\n7	1	32	21	644	6\n7	1	33	31\n7	1	33	2\n7	1	37	31	32\n7	1	37	502	2273\n7	1	38	32	33\n7	1	38	11932	11627\n7	1	39	33	34\n7	1	39	21	6\n7	1	44	32	33	34\n7	1	44	24	1130	4390\n7	1	45	32	33	34\n7	1	45	33	3321	245\n7	2	22	16\n7	2	22	1\n7	2	28	21	22\n7	2	28	1	5\n7	2	29	22	23\n7	2	29	324	71\n7	2	30	22	23\n7	2	30	2	650\n7	2	31	22	23	24\n7	2	31	502	6	2\n7	2	32	22	23\n7	2	32	818	13444\n7	2	33	22	23	24\n7	2	33	114	13722	2263\n7	2	34	23	24\n7	2	34	4635	6\n7	3	16	12\n7	3	16	1\n7	3	21	16\n7	3	21	1\n7	3	22	16	17\n7	3	22	1603	162\n7	3	23	16	17\n7	3	23	6704	25824\n7	3	24	17\n7	3	24	2271\n7	4	12	8\n7	4	12	1\n7	4	16	10	11	12\n7	4	16	31	5562	2715\n7	4	17	10	11	12	13\n7	4	17	1	16302	11950	4\n7	5	8	5\n7	5	8	1\n7	5	10	5	6\n7	5	10	16	16\n7	5	11	5	6	7	8\n7	5	11	1041	14423	6365	35\n7	5	12	5	6	7	8	9\n7	5	12	3	781	8051	5829	1\n7	5	13	8\n7	5	13	4\n7	6	5	1\n7	6	5	1061\n7	6	6	1	2\n7	6	6	14745	475\n7	6	7	1	2	3	4\n7	6	7	1520	10167	2624	105\n7	6	8	2	3	4\n7	6	8	1	4431	1436\n7	6	9	4\n7	6	9	1\n8	1	28	20\n8	1	28	1\n8	1	29	27\n8	1	29	16\n8	1	30	27	28\n8	1	30	34	119\n8	1	31	28	29\n8	1	31	49	68\n8	1	36	29	30\n8	1	36	30	294\n8	1	37	30	31	32\n8	1	37	1681	4569	92\n8	1	38	31	32\n8	1	38	152	469\n8	1	44	31	32\n8	1	44	866	1075\n8	1	45	32\n8	1	45	535\n8	2	20	14\n8	2	20	1\n8	2	27	20	21\n8	2	27	15	35\n8	2	28	21	22\n8	2	28	93	75\n8	2	29	20	21	22\n8	2	29	19	11	68\n8	2	30	21	22\n8	2	30	1809	166\n8	2	31	21	22\n8	2	31	617	4970\n8	2	32	22	23\n8	2	32	1798	373\n8	3	14	11\n8	3	14	1\n8	3	20	15\n8	3	20	34\n8	3	21	15	16\n8	3	21	624	1941\n8	3	22	16	17\n8	3	22	6725	352\n8	3	23	17\n8	3	23	373\n8	4	11	8\n8	4	11	1\n8	4	15	11	12\n8	4	15	504	154\n8	4	16	11	12\n8	4	16	343	8323\n8	4	17	12	13\n8	4	17	721	4\n8	5	8	6\n8	5	8	1\n8	5	11	7	8	9\n8	5	11	33	801	13\n8	5	12	7	8	9\n8	5	12	2	7305	1891\n8	5	13	8	9\n8	5	13	3	1\n8	6	6	3\n8	6	6	1\n8	6	7	3	4\n8	6	7	4	31\n8	6	8	3	4	5	6\n8	6	8	21	4472	3600	16\n8	6	9	4	5	6\n8	6	9	5	1569	331\n8	7	3	1\n8	7	3	26\n8	7	4	1\n8	7	4	4508\n8	7	5	1	2	3\n8	7	5	2754	2400	15\n8	7	6	2	3\n8	7	6	327	20\n9	1	28	25	26\n9	1	28	9	21\n9	1	29	26	27\n9	1	29	28	40\n9	1	30	27\n9	1	30	3\n9	1	35	27	28\n9	1	35	9	89\n9	1	36	28	29	30\n9	1	36	474	1257	123\n9	1	37	29	30\n9	1	37	38	342\n9	1	44	28	29	30	31\n9	1	44	24	276	499	14\n9	2	25	18	19\n9	2	25	1	8\n9	2	26	19	20\n9	2	26	28	21\n9	2	27	19	20	21\n9	2	27	9	37	6\n9	2	28	19	20\n9	2	28	383	204\n9	2	29	19	20	21\n9	2	29	92	1204	275\n9	2	30	20	21\n9	2	30	243	721\n9	2	31	21\n9	2	31	14\n9	3	18	14\n9	3	18	1\n9	3	19	13	14	15\n9	3	19	3	513	4\n9	3	20	14	15\n9	3	20	271	1438\n9	3	21	15	16\n9	3	21	853	163\n9	4	13	10\n9	4	13	3\n9	4	14	10	11\n9	4	14	87	698\n9	4	15	11	12\n9	4	15	1919	376\n9	4	16	12\n9	4	16	163\n9	5	10	8\n9	5	10	90\n9	5	11	8	9\n9	5	11	2191	426\n9	5	12	8	9\n9	5	12	30	509\n9	6	8	5	6\n9	6	8	159	2152\n9	6	9	5	6\n9	6	9	3	932\n9	7	5	2	3\n9	7	5	144	18\n9	7	6	2	3	4\n9	7	6	198	2462	424\n9	8	2	1\n9	8	2	342\n9	8	3	1\n9	8	3	2480\n9	8	4	1	2\n9	8	4	413	11\n10	1	26	23\n10	1	26	1\n10	1	27	24	25\n10	1	27	10	3\n10	1	28	25\n10	1	28	15\n10	1	34	25	26\n10	1	34	6	23\n10	1	35	26	27	28\n10	1	35	146	387	59\n10	1	36	27	28\n10	1	36	7	102\n10	1	43	26	27\n10	1	43	15	31\n10	1	44	26	27	28\n10	1	44	2	49	139\n10	2	23	17\n10	2	23	1\n10	2	24	17	18\n10	2	24	4	6\n10	2	25	17	18	19\n10	2	25	6	7	11\n10	2	26	17	18\n10	2	26	81	105\n10	2	27	17	18	19\n10	2	27	9	314	151\n10	2	28	18	19\n10	2	28	31	269\n10	3	17	12	13\n10	3	17	49	52\n10	3	18	13	14\n10	3	18	452	11\n10	3	19	13	14\n10	3	19	11	420\n10	4	12	9\n10	4	12	49\n10	4	13	9	10\n10	4	13	60	455\n10	4	14	10	11\n10	4	14	314	117\n10	5	9	7\n10	5	9	109\n10	5	10	7	8\n10	5	10	150	619\n10	5	11	8\n10	5	11	117\n10	6	7	5	6\n10	6	7	143	116\n10	6	8	6\n10	6	8	736\n10	7	5	3	4\n10	7	5	1	142\n10	7	6	3	4\n10	7	6	6	846\n10	8	3	1	2\n10	8	3	6	1\n10	8	4	1	2\n10	8	4	122	866\n10	9	1	1\n10	9	1	128\n10	9	2	1\n10	9	2	867\n11	1	25	22\n11	1	25	2\n11	1	26	23\n11	1	26	9\n11	1	27	24\n11	1	27	1\n11	1	33	24\n11	1	33	9\n11	1	34	24	25	26\n11	1	34	61	146	21\n11	1	35	25	26\n11	1	35	2	17\n11	1	43	24	25	26\n11	1	43	7	30	29\n11	2	22	15\n11	2	22	2\n11	2	23	16	17\n11	2	23	7	2\n11	2	24	15	16	17\n11	2	24	26	51	1\n11	2	25	16	17\n11	2	25	107	71\n11	2	26	17	18\n11	2	26	66	1\n11	3	15	11\n11	3	15	28\n11	3	16	11	12\n11	3	16	105	60\n11	3	17	12	13\n11	3	17	138	2\n11	3	18	13\n11	3	18	1\n11	4	11	8	9\n11	4	11	129	4\n11	4	12	8	9\n11	4	12	1	197\n11	4	13	9	10\n11	4	13	1	2\n11	5	8	6\n11	5	8	130\n11	5	9	6	7\n11	5	9	5	197\n11	5	10	7\n11	5	10	2\n11	6	6	4	5\n11	6	6	6	129\n11	6	7	5	6\n11	6	7	198	1\n11	7	4	3\n11	7	4	6\n11	7	5	3	4\n11	7	5	92	235\n11	7	6	4\n11	7	6	1\n11	8	3	2\n11	8	3	98\n11	8	4	2	3\n11	8	4	235	1\n11	9	2	1\n11	9	2	333\n11	9	3	1\n11	9	3	1\n11	10	1	1\n11	10	1	334\n12	1	25	21	22\n12	1	25	1	2\n12	1	33	22	23	24\n12	1	33	14	48	5\n12	1	34	23	24\n12	1	34	1	18\n12	1	43	22	23	24\n12	1	43	2	8	19\n12	2	21	14\n12	2	21	1\n12	2	22	14	15\n12	2	22	16	2\n12	2	23	14	15\n12	2	23	30	27\n12	2	24	15	16\n12	2	24	41	1\n12	3	14	9	10\n12	3	14	12	35\n12	3	15	10	11\n12	3	15	45	25\n12	3	16	11\n12	3	16	1\n12	4	9	6	7\n12	4	9	3	9\n12	4	10	7	8\n12	4	10	64	16\n12	4	11	8\n12	4	11	26\n12	5	6	5\n12	5	6	3\n12	5	7	5	6\n12	5	7	54	19\n12	5	8	6\n12	5	8	42\n12	6	5	3	4\n12	6	5	1	56\n12	6	6	4	5\n12	6	6	56	5\n12	7	3	3\n12	7	3	1\n12	7	4	3\n12	7	4	112\n12	7	5	3\n12	7	5	5\n12	8	3	2\n12	8	3	118\n12	9	2	1\n12	9	2	118\n12	10	1	1\n12	10	1	118\n12	11	1	1\n12	11	1	118\n13	1	23	19	20\n13	1	23	1	1\n13	1	24	20\n13	1	24	1\n13	1	32	21\n13	1	32	3\n13	1	33	21	22\n13	1	33	14	19\n13	1	43	21	22\n13	1	43	3	7\n13	2	19	12\n13	2	19	1\n13	2	20	13\n13	2	20	2\n13	2	21	12	13\n13	2	21	4	16\n13	2	22	13	14\n13	2	22	21	5\n13	3	12	8\n13	3	12	5\n13	3	13	8	9\n13	3	13	7	32\n13	3	14	9\n13	3	14	5\n13	4	8	5	6\n13	4	8	2	10\n13	4	9	6	7\n13	4	9	35	2\n13	5	5	4\n13	5	5	2\n13	5	6	4	5\n13	5	6	32	13\n13	5	7	5\n13	5	7	2\n13	6	4	3\n13	6	4	34\n13	6	5	3	4\n13	6	5	12	3\n13	7	3	2	3\n13	7	3	42	4\n13	7	4	3\n13	7	4	3\n13	8	2	1	2\n13	8	2	8	34\n13	8	3	2\n13	8	3	7\n13	9	1	1\n13	9	1	8\n13	9	2	1\n13	9	2	41\n13	10	1	1\n13	10	1	49\n13	11	1	1\n13	11	1	49\n13	12	1	1\n13	12	1	49\n14	1	23	18\n14	1	23	1\n14	1	32	19	20\n14	1	32	1	10\n14	1	33	21\n14	1	33	2\n14	1	43	20\n14	1	43	2\n14	2	18	11\n14	2	18	1\n14	2	19	11\n14	2	19	1\n14	2	20	11	12\n14	2	20	5	7\n14	2	21	12\n14	2	21	2\n14	3	11	7\n14	3	11	7\n14	3	12	7	8\n14	3	12	2	7\n14	4	7	5\n14	4	7	9\n14	4	8	5\n14	4	8	7\n14	5	5	3	4\n14	5	5	11	5\n14	6	3	2\n14	6	3	11\n14	6	4	2	3\n14	6	4	1	4\n14	7	2	1	2\n14	7	2	6	6\n14	7	3	2\n14	7	3	4\n14	8	1	1\n14	8	1	6\n14	8	2	1\n14	8	2	10\n14	9	1	1\n14	9	1	16\n14	10	1	1\n14	10	1	16\n14	11	1	1\n14	11	1	16\n14	12	1	1\n14	12	1	16\n14	13	1	1\n14	13	1	16\n15	1	32	19\n15	1	32	5\n15	1	43	19	20\n15	1	43	2	1\n15	2	19	10	11\n15	2	19	2	5\n15	2	20	11\n15	2	20	1\n15	3	10	6\n15	3	10	2\n15	3	11	6	7\n15	3	11	1	5\n15	4	6	4\n15	4	6	3\n15	4	7	4\n15	4	7	5\n15	5	4	2	3\n15	5	4	2	6\n15	6	2	1	2\n15	6	2	1	1\n15	6	3	2\n15	6	3	6\n15	7	1	1\n15	7	1	1\n15	7	2	1\n15	7	2	7\n15	8	1	1\n15	8	1	8\n15	9	1	1\n15	9	1	8\n15	10	1	1\n15	10	1	8\n15	11	1	1\n15	11	1	8\n15	12	1	1\n15	12	1	8\n15	13	1	1\n15	13	1	8\n15	14	1	1\n15	14	1	8\n16	1	32	18\n16	1	32	1\n16	2	18	10\n16	2	18	1\n16	3	10	6\n16	3	10	1\n16	4	6	4\n16	4	6	1\n16	5	4	2\n16	5	4	1\n16	6	2	1\n16	6	2	1\n16	7	1	1\n16	7	1	1\n16	8	1	1\n16	8	1	1\n16	9	1	1\n16	9	1	1\n16	10	1	1\n16	10	1	1\n16	11	1	1\n16	11	1	1\n16	12	1	1\n16	12	1	1\n16	13	1	1\n16	13	1	1\n16	14	1	1\n16	14	1	1\n16	15	1	1\n16	15	1	1\n17	1	33	18\n17	1	33	1\n17	2	18	10\n17	2	18	1\n17	3	10	6\n17	3	10	1\n17	4	6	3\n17	4	6	1\n17	5	3	2\n17	5	3	1\n17	6	2	1\n17	6	2	1\n17	7	1	1\n17	7	1	1\n17	8	1	1\n17	8	1	1\n17	9	1	1\n17	9	1	1\n17	10	1	1\n17	10	1	1\n17	11	1	1\n17	11	1	1\n17	12	1	1\n17	12	1	1\n17	13	1	1\n17	13	1	1\n17	14	1	1\n17	14	1	1\n17	15	1	1\n17	15	1	1\n17	16	1	1\n17	16	1	1\n18	1	33	17\n18	1	33	1\n18	2	17	9\n18	2	17	1\n18	3	9	5\n18	3	9	1\n18	4	5	3\n18	4	5	1\n18	5	3	1\n18	5	3	1\n18	6	1	1\n18	6	1	1\n18	7	1	1\n18	7	1	1\n18	8	1	1\n18	8	1	1\n18	9	1	1\n18	9	1	1\n18	10	1	1\n18	10	1	1\n18	11	1	1\n18	11	1	1\n18	12	1	1\n18	12	1	1\n18	13	1	1\n18	13	1	1\n18	14	1	1\n18	14	1	1\n18	15	1	1\n18	15	1	1\n18	16	1	1\n18	16	1	1\n18	17	1	1\n18	17	1	1\n21	1	35	16\n21	1	35	1\n21	2	16	8\n21	2	16	1\n21	3	8	4\n21	3	8	1\n21	4	4	2\n21	4	4	1\n21	5	2	1\n21	5	2	1\n21	6	1	1\n21	6	1	1\n21	7	1	1\n21	7	1	1\n21	8	1	1\n21	8	1	1\n21	9	1	1\n21	9	1	1\n21	10	1	1\n21	10	1	1\n21	11	1	1\n21	11	1	1\n21	12	1	1\n21	12	1	1\n21	13	1	1\n21	13	1	1\n21	14	1	1\n21	14	1	1\n21	15	1	1\n21	15	1	1\n21	16	1	1\n21	16	1	1\n21	17	1	1\n21	17	1	1\n21	18	1	1\n21	18	1	1\n21	19	1	1\n21	19	1	1\n21	20	1	1\n21	20	1	1\n28	1	39	15\n28	1	39	1\n28	2	15	7\n28	2	15	1\n28	3	7	4\n28	3	7	1\n28	4	4	2\n28	4	4	1\n28	5	2	1\n28	5	2	1\n28	6	1	1\n28	6	1	1\n28	7	1	1\n28	7	1	1\n28	8	1	1\n28	8	1	1\n28	9	1	1\n28	9	1	1\n28	10	1	1\n28	10	1	1\n28	11	1	1\n28	11	1	1\n28	12	1	1\n28	12	1	1\n28	13	1	1\n28	13	1	1\n28	14	1	1\n28	14	1	1\n28	15	1	1\n28	15	1	1\n28	16	1	1\n28	16	1	1\n28	17	1	1\n28	17	1	1\n28	18	1	1\n28	18	1	1\n28	19	1	1\n28	19	1	1\n28	20	1	1\n28	20	1	1\n28	21	1	1\n28	21	1	1\n28	22	1	1\n28	22	1	1\n28	23	1	1\n28	23	1	1\n28	24	1	1\n28	24	1	1\n28	25	1	1\n28	25	1	1\n28	26	1	1\n28	26	1	1\n28	27	1	1\n28	27	1	1\n31	1	41	14\n31	1	41	2\n31	2	14	7\n31	2	14	2\n31	3	7	3\n31	3	7	2\n31	4	3	1\n31	4	3	2\n31	5	1	1\n31	5	1	2\n31	6	1	1\n31	6	1	2\n31	7	1	1\n31	7	1	2\n31	8	1	1\n31	8	1	2\n31	9	1	1\n31	9	1	2\n31	10	1	1\n31	10	1	2\n31	11	1	1\n31	11	1	2\n31	12	1	1\n31	12	1	2\n31	13	1	1\n31	13	1	2\n31	14	1	1\n31	14	1	2\n31	15	1	1\n31	15	1	2\n31	16	1	1\n31	16	1	2\n31	17	1	1\n31	17	1	2\n31	18	1	1\n31	18	1	2\n31	19	1	1\n31	19	1	2\n31	20	1	1\n31	20	1	2\n31	21	1	1\n31	21	1	2\n31	22	1	1\n31	22	1	2\n31	23	1	1\n31	23	1	2\n31	24	1	1\n31	24	1	2\n31	25	1	1\n31	25	1	2\n31	26	1	1\n31	26	1	2\n31	27	1	1\n31	27	1	2\n31	28	1	1\n31	28	1	2\n31	29	1	1\n31	29	1	2\n31	30	1	1\n31	30	1	2\n"
#define READ_LEN_DIST "37	38	39	40	41	42	43	44	45	46	47	48	49	50	51	52	53	54	55	56	57	58	59	60	61	62	63	64	65	66	67	68	69	70	71	72	73	74	75	76	77	78	79	80	81	82	83	84	85	86	87	88	89	90	91	92	93	94	95	96	97	98	99	100	101	102	103	104	105	106	107	108	109	110	111	112	113	114	115	116	117	118	119	120	121	122	123	124	125	126	127	128	129	130	131	132	133	134	135	136	137	138	139	140	141	142	143	144	145	146	147	148	149	150	151	152	153	154	155	156	157	158	159	160	161	162	163	164	165	166	167	168	169	170	171	172	173	174	175	176	177	178	179	180	181	182	183	184	185	186	187	188	189	190	191	192	193	194	195	196	197	198	199	200	201	202	203	204	205	206	207	208	209	210	211	212	213	214	215	216	217	218	219	220	221	222	223	224	225	226	227	228	229	230	231	232	233	234	235	236	237	238	239	240	241	242	243	244	245	246	247	248	249	250	251	252	253	254	255	256	257	258	259	260	261	262	263	264	265	266	267	268	269	270	271	272	273	274	275	276	277	278	279	280	281	282	283	284	285	286	287	288	289	290	291	292	293	294	295	296	297	298	299	300	301	302	303	304	305	306	307	308	309	310	311	312	313	314	315	316	317	318	319	320	321	322	323	324	325	326	327	328	330	331	332	333	334	335	337	347	385\n3	4	2	6	2	13	12	20	32	46	47	68	74	79	86	98	116	125	125	154	163	161	169	164	153	166	145	151	138	141	145	158	145	154	151	141	148	155	142	167	169	171	165	201	188	202	174	202	228	231	210	249	257	238	235	278	235	246	301	360	307	311	369	390	395	380	445	431	460	506	539	549	563	586	641	717	654	720	757	730	760	851	868	932	925	1045	1099	1138	1122	1223	1187	1305	1390	1383	1517	1518	1599	1645	1757	1718	1786	1895	1971	2027	2125	2015	2161	2154	2297	2259	2436	2489	2571	2522	2732	2668	2809	2822	2821	2897	3061	2954	3117	3131	3263	3173	3237	3302	3255	3410	3413	3491	3536	3539	3568	3630	3495	3648	3683	3726	3714	3698	3749	3864	3871	3763	3818	3835	3890	3784	3834	4059	3921	4051	3957	4075	4112	4011	3955	4007	3984	4070	4006	3997	4176	4050	3958	3895	3956	4061	3969	4076	3967	4069	3946	3956	4030	4051	4130	4125	4218	4101	4080	4091	4157	4184	4266	4206	4302	4622	4771	4947	5115	5463	5650	5936	6564	6610	7219	7709	7995	8681	9148	9907	10420	11186	11775	12429	13129	13416	14180	14586	14937	15190	15751	15616	15979	16196	16169	16084	15725	15424	15109	14763	14149	13729	13130	12368	12055	11187	10365	9921	9024	8450	7676	6953	6424	5871	5264	4903	4285	3741	3427	3007	2700	2427	2113	1845	1553	1458	1341	1099	999	901	787	692	636	527	476	427	416	335	286	263	233	176	183	185	123	107	115	87	74	59	42	32	24	38	27	26	26	12	18	13	13	9	10	8	7	8	3	4	5	3	3	1	1	1	1	1	1"
#define FLX_CYCLE_NUM 100
using namespace std;

inline double r_prob(){
    return rand()/(double)RAND_MAX;
};

inline char rand_base(){
    short base=(short)ceil(r_prob()*5);
    switch(base){
            case 1:
                return 'a';
            case 2:
                return 'c';
            case 3:
                return 'g';
            case 4:
                return 't'; 
            case 5:
                return 'n'; 
    }
};

class read_profile{
public:
    void default_profile();
    void user_profile(string qual_profile, string mc_qual_profile, string indel_erro_profile, string length_profile);
    static double prob_err[HIGHEST_QUAL];
    static void set_err_prob(){
        for(int i=0; i<HIGHEST_QUAL; i++){
            prob_err[i]=pow(10,-i/(double)10);
        }
    };


    static const unsigned int max_dist_number=1000000; //need >100000
    bool read_emp_dist(string infile, vector< map <unsigned int, unsigned short> >& qual_dist);
    bool read_emp_dist(istream& input,vector< map <unsigned int, unsigned short> >& qual_dist);
    bool read_emp_dist(string infile, map <unsigned int, unsigned int>& dist);
    bool read_emp_dist(istream& input, map <unsigned int, unsigned int> & dist);
    
    bool read_emp_dist(string infile, map<string, map <unsigned int, unsigned short> >& qual_dist, short num_id_field);
    bool read_emp_dist(istream& input, map<string, map<unsigned int, unsigned short> >& qual_dist, short num_id_field);

    void read_indel_error(string error_profile_file);

    //assign 454 read qual. the read_seq may change imposed by maxnum cycle FLX_CYCLE_NUM
    bool get_read_qual(string& read_seq, string& seq, vector<string>&aln, vector<short>& qual, short cycle=FLX_CYCLE_NUM); 
    bool get_read_qual_fast(vector<int>& lenVec, unsigned long b_pos, string& read_seq, string& seq, vector<string>&aln, vector<short>& qual, short& start_cyc, short cycle=FLX_CYCLE_NUM);

    bool get_read_qual(vector<short>& read_qual, bool first=true);
    bool get_read_qual(vector<short>& read_qual, int len, bool first=true);
    //TODO
    //bool read_conti_dist(); 
    ~read_profile();
    vector< map <unsigned int, unsigned short> > qual_1stbase_dist; //first postion
    map<string, map <unsigned int, unsigned short> > qual_dist_mc; //remaining positon in Markov chain
    map<unsigned int, unsigned int> read_len_dist;
    map<unsigned int, unsigned int> fragment_length;
    int get_ran_flagment_len(){
        if(fragment_length.size()==0) return 0;
        int cc=(int) ceil(r_prob()*max_dist_number);
        map<unsigned int, unsigned int>::iterator it=fragment_length.lower_bound(cc);
        if(it==fragment_length.end()) return 0;
        return it->second; 
    };
    int get_ran_read_len(){
        if(read_len_dist.size()==0) return 0;
        int cc=(int) ceil(r_prob()*max_dist_number);
        map<unsigned int, unsigned int>::iterator it=read_len_dist.lower_bound(cc);
        if(it==read_len_dist.end()) return 0;
        return it->second; 
    };
    vector<double> under_call_error;
    vector<double> over_call_error;
    vector<double> total_error;
//    vector<double> under2total_ratio;
private:
    //int max_len_pair_first;
    //int max_len_pair_second;
    bool get_read_qual(vector< map <unsigned int, unsigned short> >&qual_dist, vector<short>& read_qual, int len);

};
